/** @file   animplayer.h
 * @brief   Declarations of AnimPlayer - class.
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:41 $
 * @author  Tomi Lamminsaari
 */
 
#ifndef H_WWW_ANIMSTORAGE_H
#define H_WWW_ANIMSTORAGE_H

#include <list>
#include "eng2d.h"


namespace WeWantWar {



/** @class  AnimPlayer
 * @brief   Manages the background animations like explosions.
 * @author  Tomi Lamminsaari
 *
 * AnimPlayer - stores and updates the animation we spawn during the
 * gameplay. This class has only static members.
 * 
 * Use <code>spawn</code>- or <code>spawnLit</code>-methods to add new
 * animations to the animationtable. If the animation is not loopped and
 * it has been played through, the <code>update()</code>-method destroyed
 * the animation from the table.
 *
 * The animations need to have the RLE_SPRITE - pointer assigned before
 * you are allowed to add them to AnimPlayer
 */
class AnimPlayer
{
public:
  ///
  /// Static members and methods
  /// ==========================
  
  /** Maximum number of simultaneus animations */
  static const int MAX_ANIMS = 64;
  
  /** Datatype for animations drawing modes */
  typedef int DrawMode;
  /** Normal drawing mode. No transparency. */
  static const DrawMode DMODE_NORMAL = 0;
  /** Drawing with burn-blender function */
  static const DrawMode DMODE_BURN = 1;
  /** Drawing with dodge-blender function */
  static const DrawMode DMODE_DODGE = 2;
  /** Drawing mode for normal trans-blender */
  static const DrawMode DMODE_TRANS = 3;
  /** Drawing mode for add-blender */
  static const DrawMode DMODE_ADD = 4;
  
  /** Id codes for special type of animations.
   */
  enum {
    /** Default id. No special meaning. */
    EDefault,
    /** A bullet shell animation. We play a sound when this animation
     * ends.
     */
    EBulletShell  = 200
  };
    
  
  /** Structure that holds the information that is needed when showing an
   * animation.
   */
  struct AnimElement {
    /** Is this normal animation or light animation. */
    DrawMode  iDrawmode;
    /** ID of the animation. See GameAnims-class to see possible values.
     * This has become quite useless at the moment.
     */
    int       iType;
    /** The alpha value we pass to the blender functions. If drawing mode
     * is DMODE_NORMAL this value has no effect.
     */
    int       iAlpha;
    /** Position of this animation (pixels from topleft corner of the map) */
    eng2d::Vec2D      iPos;
    /** Movement vector. Animation position will be updated by this vector. */
    eng2d::Vec2D      iMovement;
    /** The animation itself. */
    eng2d::Animation  iAnim;
    /** Uid of this animation */
    int iUid;
  };
  
  /** The table that holds all the animations */
  static std::list<AnimElement*> m_table;
  
  
  /** Initializes this AnimPlayer. */
  static void init();
  /** Cleans up the animations. */
  static void cleanup();
  
  /** Spawns new animation.
   * @param     a                 The animation itself. This makes copy of this
   *                              animation and deletes it when it's played
   *                              through unless it's loopped.
   * @param     p                 Position
   * @param     t                 ID of the animation. See GameAnims - class.
   * @return    Uid of the animation.
   */
  static int spawn( const eng2d::Animation& a, const eng2d::Vec2D& p, int t = 0 );
  
  /** Spawns new moving animation.
   * @param     a                 The animation.
   * @param     p                 Position
   * @param     m                 Movement vector
   * @param     t                 ID of the animation.
   * @return    Uid of the animation.
   */
  static int spawn( const eng2d::Animation& a, const eng2d::Vec2D& p,
                     const eng2d::Vec2D& m, int t );
  
  /** Spawns light-animation. Light-animations differs from normal animations
   * since they make the background lighter. These are used for instance in
   * weapons firing flames.
   * @param     a                 The animation itself. This makes copy of this
   *                              animation and deletes it when it's played
   *                              through unless it's loopped.
   * @param     p                 Position
   * @param     t                 ID of the animation. See GameAnims - class.
   * @return    Uid of the animation.
   */
  static int spawnLit( const eng2d::Animation& a, const eng2d::Vec2D& p,
                        int t );

  /** Spawns an animation with given parameters.
   * @param     rAnim             Reference to the animation itself.
   * @param     rP                The center position of the animation.
   * @param     dmode             Drawing mode.
   * @param     alpha             Alpha-value. Now meaning if drawing mode
   *                              is DMODE_NORMAL
   * @param     rMov              The movement vector.
   * @return    Uid of the animation.
   */
  static int spawn( const eng2d::Animation& rAnim, const eng2d::Vec2D& rP,
                     DrawMode dmode, int alpha, const eng2d::Vec2D& rMov );
                     
  /** Updates the animations. This also deletes the animations that reach
   * the end unless they're loopped.
   */
  static void update();
  
  /** Draws the animations on the given bitmap.
   * @param     pB                Pointer to target bitmap.
   */
  static void redraw( BITMAP* pB );
  
  /** Returns the index'th animation.
   * @param     index             Index of the animation.
   * @return    Pointer to Index'th animation.
   */
  static AnimElement* getAnim(int index);

  /** Returns the animation that matches with given uid.
   * @param     aUid            Uid of the animation.
   * @return    Pointer to animation or nullpointer if no animation was found.
   */
  static AnimElement* findAnim( int aUid );

  /** Removes the animation whose uid is given.
   * @param     aUid            Uid of the animation being removed.
   */
  static void removeAnimation( int aUid );
  

  /** Constructor.
   */
  AnimPlayer();
  
  /** Copy constructor.
   */
  AnimPlayer(const AnimPlayer& rO);
  
  /** Destructor.
   */
  ~AnimPlayer();
  
  /** Assingment operator
   */
  AnimPlayer& operator = (const AnimPlayer& rO);

  /** Public methods **/


  /** Public GET-methods **/

private:

  /** Private methods **/
  
  /** This keeps track of the uids we give to the animations. */
  static int uidCounter;
  
  
  /** Does the actions that are needed when special animation has been
   * played through. Usually this means that we play a sample of some
   * kind.
   * @param     aAnimElement      Pointer to the animation.
   */
  static void animationEndingActions( AnimElement* aAnimElement );
  


  /** Members **/

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: animplayer.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.1  2005-12-30 23:46:38+02  lamminsa
 * Uids added.
 *
 * Revision 1.0  2005-11-06 01:16:47+02  lamminsa
 * Initial revision
 *
 */
